package xim.math

import xim.poc.UiElementHelper
import xim.poc.browser.LocalStorage
import kotlin.math.sqrt

data class Vector2f(var x: Float = 0f, var y: Float = 0f) {

    fun copyFrom(other: Vector2f): Vector2f {
        x = other.x
        y = other.y
        return this
    }

    companion object {
        val ZERO = Vector2f(0f, 0f)
        val ONE = Vector2f(1f, 1f)

        fun distance(a: Vector2f, b: Vector2f): Float {
            return sqrt(distanceSquared(a, b))
        }

        fun distanceSquared(a: Vector2f, b: Vector2f) : Float {
            val dX = a.x - b.x
            val dY = a.y - b.y
            return dX*dX + dY*dY
        }

        fun screenSpaceToUiSpace(v: Vector2f): Vector2f {
            val resolution = LocalStorage.getConfiguration().screenSettings.resolution
            return v * UiElementHelper.globalUiScale * (1f / resolution)
        }

        fun uiSpaceToScreenSpace(v: Vector2f): Vector2f {
            val resolution = LocalStorage.getConfiguration().screenSettings.resolution
            return v * resolution / UiElementHelper.globalUiScale
        }

    }

    operator fun plus(other: Vector2f) : Vector2f {
        return Vector2f(x + other.x, y + other.y)
    }

    operator fun minus(other: Vector2f) : Vector2f {
        return Vector2f(x - other.x, y - other.y)
    }

    operator fun times(value: Float) : Vector2f {
        return Vector2f(x*value, y*value)
    }

    operator fun times(other: Vector2f) : Vector2f {
        return Vector2f(x*other.x, y*other.y)
    }

    operator fun div(other: Vector2f): Vector2f {
        return Vector2f(x/other.x, y/other.y)
    }

    operator fun get(uv: Axis): Float {
        return when(uv) {
            Axis.X -> x
            Axis.Y -> y
            else -> throw IllegalStateException("illegal axis on 2d vector")
        }
    }

    operator fun set(uv: Axis, value: Float) {
        when(uv) {
            Axis.X -> x = value
            Axis.Y -> y = value
            else -> throw IllegalStateException("illegal axis on 2d vector")
        }
    }

    fun scale(scale: Vector2f): Vector2f {
        return Vector2f(x * scale.x, y * scale.y)
    }

}