package xim.poc.browser

import web.dom.document
import web.gamepad.Gamepad
import web.gamepad.GamepadMappingType
import web.html.HTMLOptionElement
import web.html.HTMLSelectElement
import web.navigator.navigator
import xim.math.Vector2f
import xim.poc.browser.Keyboard.KeyState.*
import kotlin.math.abs

object GamepadManager {

    private val activeGamepads = LinkedHashMap<String, GamepadState>()

    private var enabled = false
    private var flipped = false

    private val select by lazy { document.getElementById("gamepadId") as HTMLSelectElement }

    fun poll() {
        enabled = checkEnabled()
        if (!enabled) { return }

        flipped = checkFlipped()

        val currentGamePads = navigator.getGamepads()
            .filterNotNull()
            .filter { it.connected && it.mapping == GamepadMappingType.standard }

        for (gamepad in currentGamePads) {
            val internalGamepad = activeGamepads.getOrPut(gamepad.id) {
                addOption(gamepad)
                GamepadState(gamepad)
            }

            internalGamepad.poll(gamepad)
        }

        val itr = activeGamepads.iterator()
        while (itr.hasNext()) {
            val next = itr.next()
            if (currentGamePads.any { it.id == next.key }) { continue }

            removeOption(next.key)
            itr.remove()
        }
    }

    fun isActive(): Boolean {
        return enabled && getActiveGamepad() != null
    }

    fun isFlipped(): Boolean {
        return flipped
    }

    fun isPressed(gameKey: GameKey): Boolean {
        if (!enabled) { return false }

        val gamepad = getActiveGamepad() ?: return false
        if (anyHotbarMetaIsActive(gamepad)) { return false }
        return gamepad.isPressed(gameKey)
    }

    fun isPressedOrRepeated(gameKey: GameKey): Boolean {
        if (!enabled) { return false }

        val gamepad = getActiveGamepad() ?: return false
        if (anyHotbarMetaIsActive(gamepad)) { return false }
        return gamepad.isPressedOrRepeated(gameKey)
    }

    fun isHotBarActive(hotbarIndex: Int): Boolean {
        if (!enabled) { return false }

        val gamepad = getActiveGamepad() ?: return false
        val button = hotbarMetaMapping(hotbarIndex) ?: return false
        return gamepad.isPressedOrRepeated(button)
    }

    fun isHotbarPressed(hotbarIndex: Int, hotbarEntry: Int): Boolean {
        if (!enabled) { return false }

        val gamepad = getActiveGamepad() ?: return false
        val hotbarMetaButton = hotbarMetaMapping(hotbarIndex) ?: return false

        if (!gamepad.isPressedOrRepeated(hotbarMetaButton)) { return false }

        val hotbarEntryButton = (if (flipped) {
            flippedHotbarMapping(hotbarEntry)
        } else {
            hotbarMapping(hotbarEntry)
        }) ?: return false

        return gamepad.isPressed(hotbarEntryButton)
    }

    fun getLeftAxisInput(): Vector2f? {
        if (!enabled) { return null }
        val gamepad = getActiveGamepad() ?: return null
        return applyDeadZone(gamepad.leftAxis)
    }

    fun getRightAxisInput(): Vector2f? {
        if (!enabled) { return null }
        val gamepad = getActiveGamepad() ?: return null
        return applyDeadZone(gamepad.rightAxis)
    }

    private fun getActiveGamepad(): GamepadState? {
        return activeGamepads[select.value]
    }

    private fun applyDeadZone(axisInput: Vector2f): Vector2f {
        if (abs(axisInput.x) < 0.1f ) { axisInput.x = 0f }
        if (abs(axisInput.y) < 0.1f ) { axisInput.y = 0f }
        return axisInput
    }

    private fun anyHotbarMetaIsActive(gamepad: GamepadState): Boolean {
        return gamepad.isPressedOrRepeated(4) ||
                gamepad.isPressedOrRepeated(6) ||
                gamepad.isPressedOrRepeated(7)
    }

    private fun checkEnabled(): Boolean {
        return LocalStorage.getConfiguration().gamepadSettings.enabled
    }

    private fun checkFlipped(): Boolean {
        return LocalStorage.getConfiguration().gamepadSettings.flipped
    }

    private fun addOption(gamepad: Gamepad) {
        for (option in select.options) {
            if (option.value == gamepad.id) { return }
        }

        val option = document.createElement("option") as HTMLOptionElement
        option.value = gamepad.id
        option.text = gamepad.id

        select.options.add(option)
        if (select.value.isBlank()) { select.value = gamepad.id }
    }

    private fun removeOption(gamepadId: String) {
        for (i in 0 until select.options.length) {
            val option = select.options[i]
            if (option.value != gamepadId) { continue }

            select.options.remove(i)
            break
        }
    }

}

private class GamepadState(gamepad: Gamepad) {

    val keyStates = Array(gamepad.buttons.size) { NONE }

    var leftAxis = Vector2f()
    var rightAxis = Vector2f()

    fun poll(gamepad: Gamepad) {
        leftAxis = Vector2f(gamepad.axes[0].toFloat(), gamepad.axes[1].toFloat())
        rightAxis = Vector2f(gamepad.axes[2].toFloat(), gamepad.axes[3].toFloat())

        for (i in 0 until gamepad.buttons.size) {
            val button = gamepad.buttons[i]
            val current = keyStates[i]

            keyStates[i] = when (current) {
                PRESSED -> if (button.pressed) { REPEATED } else { RELEASED }
                RELEASED -> if (button.pressed) { PRESSED } else { NONE }
                REPEATED -> if (button.pressed) { REPEATED } else { RELEASED }
                NONE -> if (button.pressed) { PRESSED } else { NONE }
            }
        }
    }

    fun isPressed(gameKey: GameKey): Boolean {
        val index = mapGameKeyToButton(gameKey) ?: return false
        return isPressed(index)
    }

    fun isPressed(buttonIndex: Int): Boolean {
        return keyStates.getOrNull(buttonIndex) == PRESSED
    }

    fun isPressedOrRepeated(gameKey: GameKey): Boolean {
        val index = mapGameKeyToButton(gameKey) ?: return false
        return isPressedOrRepeated(index)
    }

    fun isPressedOrRepeated(buttonIndex: Int): Boolean {
        val state = keyStates.getOrNull(buttonIndex)
        return state == PRESSED || state == REPEATED
    }

}

private fun hotbarMetaMapping(hotbarIndex: Int): Int? {
    return when (hotbarIndex) {
        0 -> 4
        1 -> 6
        2 -> 7
        else -> return null
    }
}

private fun hotbarMapping(hotbarEntry: Int): Int? {
    return when (hotbarEntry) {
        0 -> 13
        1 -> 15
        2 -> 14
        3 -> 12
        4 -> 0
        5 -> 1
        6 -> 2
        7 -> 3
        8 -> 8
        9 -> 9
        else -> null
    }
}

private fun flippedHotbarMapping(hotbarEntry: Int): Int? {
    return when (hotbarEntry) {
        0 -> 0
        1 -> 1
        2 -> 2
        3 -> 3
        4 -> 13
        5 -> 15
        6 -> 14
        7 -> 12
        8 -> 8
        9 -> 9
        else -> null
    }
}

private fun mapGameKeyToButton(gameKey: GameKey): Int? {
    return when (gameKey) {
        GameKey.UiEnter -> 0
        GameKey.UiExit -> 1
        GameKey.UiContext -> 2
        GameKey.OpenMainMenu -> 3
        GameKey.TargetCycle -> 5
        GameKey.TimeSlow -> 8
        GameKey.Pause -> 9
        GameKey.CamZoomIn -> 10
        GameKey.CamZoomOut -> 11
        GameKey.UiUp -> 12
        GameKey.UiDown -> 13
        GameKey.UiLeft -> 14
        GameKey.UiRight -> 15
        GameKey.ToggleMap -> 16
        GameKey.CamLeft -> null
        GameKey.CamRight -> null
        GameKey.CamUp -> null
        GameKey.CamDown -> null
        GameKey.MoveForward -> null
        GameKey.MoveLeft -> null
        GameKey.MoveBackward -> null
        GameKey.MoveRight -> null
        GameKey.Disengage -> null
        GameKey.TargetLock -> null
        GameKey.Rest -> null
        GameKey.FaceTarget -> null
        GameKey.Autorun -> null
        GameKey.OpenEquipMenu -> null
        GameKey.OpenInventoryMenu -> null
        GameKey.TargetSelf -> null
        GameKey.TargetParty2 -> null
        GameKey.TargetParty3 -> null
        GameKey.TargetParty4 -> null
        GameKey.TargetParty5 -> null
        GameKey.TargetParty6 -> null
        GameKey.DebugClip -> null
        GameKey.DebugGravity -> null
        GameKey.DebugSpeed -> null
        GameKey.UiConfig -> null
    }
}