package xim.poc.tools

import kotlinx.browser.document
import kotlinx.dom.clear
import org.w3c.dom.HTMLButtonElement
import org.w3c.dom.HTMLDivElement
import org.w3c.dom.HTMLInputElement
import xim.math.Vector3f
import xim.poc.*
import xim.poc.game.ActorType
import xim.poc.game.GameEngine
import xim.poc.game.event.InitialActorState

object NpcSpawningTool {

    private val input by lazy { document.getElementById("NpcSpawnId") as HTMLInputElement }
    private val button by lazy { document.getElementById("NpcSpawn") as HTMLButtonElement }
    private val next by lazy { document.getElementById("NpcSpawnNext") as HTMLButtonElement }

    private val info by lazy { document.getElementById("NpcSpawnInfo") as HTMLDivElement }
    private var currentInfo: ActorId? = null

    private val spawned = ArrayList<ActorId>()

    fun update() {
        populateInfo()
    }

    fun setup() {
        button.onclick = { spawn() }
        next.onclick = {
            spawn()

            if (input.value.startsWith("!")) {
                val value = input.value.substring(1).toIntOrNull(0x10) ?: 0x0
                input.value = "!${(value+1).toString(0x10)}"
            } else if (input.value.startsWith("^")) {
                val value = input.value.substring(1).toIntOrNull(0x10) ?: 0x0
                input.value = "^${(value+1).toString(0x10)}"
            } else {
                val currentValue = input.value.toIntOrNull(0x10) ?: 0x0
                input.value = (currentValue + 1).toString(0x10)
            }
        }
    }

    private fun spawn() {
        val rawInput = input.value
        val modelLook = if (rawInput.startsWith("!")) {
            val id = rawInput.substring(1, rawInput.length).toIntOrNull(0x10) ?: return
            ModelLook.fileTableIndex(id)
        } else if (rawInput.startsWith("^")) {
            val id = rawInput.substring(1, rawInput.length).toIntOrNull(0x10) ?: return
            ModelLook.npcWithBase(id)
        } else {
            val id = rawInput.toIntOrNull(0x10) ?: return
            ModelLook.npc(id)
        }

        spawn(modelLook)
    }

    fun spawn(modelLook: ModelLook) {
        spawned.forEach { GameEngine.submitDeleteActor(it) }
        spawned.clear()

        val position = Vector3f().copyFrom(ActorManager.player().displayPosition)
        position.x -= 2f

        val appearance = document.getElementById("appearanceInput") as HTMLInputElement
        val appearanceState = appearance.value.toIntOrNull() ?: 0

        GameEngine.submitCreateActorState(InitialActorState(
            name = "NpcTool",
            type = ActorType.Enemy,
            position = position,
            modelLook = modelLook,
            movementController = DefaultEnemyController(),
            appearanceState = appearanceState,
        )).onReady {
            spawned += it.id
        }
    }

    private fun populateInfo() {
        val current = spawned.firstNotNullOfOrNull { ActorManager[it] }
        if (current?.id == currentInfo) { return }

        if (current == null) {
            currentInfo = null
            info.clear()
            return
        }

        if (!current.isReadyToDraw()) { return }

        val model = current.actorModel?.model ?: return
        if (model is NpcWithBaseModel && model.base == null) { return }

        currentInfo = current.id
        info.clear()

        val span = document.createElement("span")
        val look = current.state.getBaseLook()
        span.innerHTML += "NPC Tool (ID: 0x${look.modelId.toString(0x10)}):<br>"

        if (model is NpcModel) {
            info.appendChild(span)
            span.innerHTML += "Model->${model.resource.resourceName}"
        } else if (model is NpcWithBaseModel && model.base != null) {
            info.appendChild(span)
            span.innerHTML += "Model->${model.resource.resourceName}<br>Base->${model.base?.resourceName}"
        }

        span.innerHTML += "<br><br>"
    }

}